#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmMaintainSpecimen;

#endregion

namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_Division : BaseTest 
	{
		private BOL.Division _testDivision;
		private DataRow _drDivision;
		private string _junkDivisionCode;

		[SetUp]
		protected void SetUp() 
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{
				DataRow drJunk = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM VamcDivision WHERE DivisionCode <> '" + this.TestDivisionCode + "' AND ServiceTypeIndicator = 1");

				_junkDivisionCode = drJunk[TABLES.VamcDivision.DivisionCode].ToString();
				_drDivision = DAL.Division.GetDivision( _junkDivisionCode );
				
				this.RefreshData = false;
			}

			this._testDivision = new Division(_junkDivisionCode);
			this._testDivision.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.ExistingSpecimen);
		}

		[Test]
		public void Ctor_DivisionCode_Pass()
		{
			Assert.IsTrue(this._testDivision!=null);
		}

		[Test]
        [ExpectedExceptionAttribute(typeof(Common.DataAccessLayerException))]
		public void Ctor_DivisionCode_Fail()
		{
			this._testDivision = new BOL.Division("X X X");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentNullException))]
		public void Ctor_DivisionCode_Null_Fail()
		{
			this._testDivision = new BOL.Division(null as string);
			Assert.IsNull(_testDivision);
		}

		[Test]
		public void Ctor_DataRow_Pass()
		{
			DataRow dr = DAL.Division.GetDivision(_junkDivisionCode).Table.NewRow();
			//Set required fields only
			dr[TABLES.VamcDivision.DivisionGuid] = Guid.NewGuid();
			dr[TABLES.VamcDivision.DivisionCode] = "XXXXX";
			dr[TABLES.VamcDivision.DivisionName] = "Unit Test 4048";
			dr[TABLES.VamcDivision.FacilityGuid] = _testDivision.FacilityGuid;
			dr[TABLES.VamcDivision.AccessionAreaId] = 999;
			dr[TABLES.VamcDivision.ServiceTypeIndicator] = false;
			dr[TABLES.VamcDivision.RecordStatusCode] = 'X';
			dr[TABLES.VamcDivision.LockInactivityTimeoutMin] = 5;
			dr[TABLES.VamcDivision.LabelPrinterUseIndicator] = false;
			dr[TABLES.VamcDivision.TimeZoneId] = 'R';
			dr[TABLES.VamcDivision.DaylightSavingsTypeId] = 2;
			//Build empty object
			BOL.Division emptyDivision = new BOL.Division(dr);
			Assert.IsNotNull(emptyDivision,"EmptyDivision");
			//
			this._testDivision = new BOL.Division(this._drDivision);
			Assert.IsNotNull(_testDivision,"LoadedDivision");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void Ctor_DataRow_Fail()
		{
			DataTable _dt = new DataTable();
			_dt.Columns.Add( "a" );
			new BOL.Division( _dt.NewRow() );
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentNullException))]
		public void Ctor_DataRow_Null_Fail()
		{
			this._testDivision = new BOL.Division( null as DataRow );
			Assert.IsNull(_testDivision);
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			DataRow dr = _testDivision.LoadDataRowFromThis( this._drDivision );
			Assert.IsTrue( dr.ItemArray.Length > 0 );
		}
		[Test]
		public void LoadDataRowFromThis_Fail()
		{
			DataRow dr = _testDivision.LoadDataRowFromThis( _drDivision.Table.NewRow() );
			Assert.IsTrue(dr.ItemArray.Length > 0);
		}

		[Test]
		public void Save_Pass()
		{
			this.RefreshData = true;

			BOL.Division div = new BOL.Division(_drDivision);
			div.IccbbaRegistrationNumber = "W1234";
			Assert.IsTrue( div.Save( Common.UpdateFunction.UnitTests ) );
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Save_Fail()
		{
		}

		[Test]
		public void ElectronicCrossmatchIndicator_Pass()
		{
			bool _electronicCrossmatchIndicator = false;
			_testDivision.ElectronicCrossmatchIndicator = _electronicCrossmatchIndicator;
			Assert.AreEqual(_electronicCrossmatchIndicator,_testDivision.ElectronicCrossmatchIndicator);
			//
			_electronicCrossmatchIndicator = true;
			_testDivision.ElectronicCrossmatchIndicator = _electronicCrossmatchIndicator;
			Assert.AreEqual(_electronicCrossmatchIndicator,_testDivision.ElectronicCrossmatchIndicator);
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void ElectronicCrossmatchIndicator_Fail()
		{
			BOL.Division testDvsn = new BOL.Division(_junkDivisionCode);
			testDvsn.ServiceTypeIndicator = false;
			testDvsn.ElectronicCrossmatchIndicator = true;
			Assert.AreEqual(false,testDvsn.ElectronicCrossmatchIndicator, "InvalidIndicator");
		}

		[Test]
		public void ServiceTypeIndicator_Get()
		{
			bool _serviceTypeIndicator = _testDivision.ServiceTypeIndicator;
			Assert.AreEqual(_testDivision.ServiceTypeIndicator,_serviceTypeIndicator);
		}

		[Test]
		public void ServiceTypeIndicator_Set()
		{
			bool _serviceTypeIndicator = false;
			_testDivision.ServiceTypeIndicator = _serviceTypeIndicator;
			Assert.AreEqual(_serviceTypeIndicator,_testDivision.ServiceTypeIndicator);
			//
			_serviceTypeIndicator = true;
			_testDivision.ServiceTypeIndicator = _serviceTypeIndicator;
			Assert.AreEqual(_serviceTypeIndicator,_testDivision.ServiceTypeIndicator);
		}

		[Test]
		public void ICCBBARegistrationNumber_Get()
		{
			string _iccbbaRegistrationNumber = _testDivision.IccbbaRegistrationNumber;
			Assert.AreEqual(_testDivision.IccbbaRegistrationNumber,_iccbbaRegistrationNumber);
		}

		[Test]
		public void ICCBBARegistrationNumber_Set_Pass()
		{
			string _iccbbaRegistrationNumber = string.Empty;
			_testDivision.IccbbaRegistrationNumber = _iccbbaRegistrationNumber;
			Assert.AreEqual(_iccbbaRegistrationNumber,_testDivision.IccbbaRegistrationNumber);

			_iccbbaRegistrationNumber = "X1234";
			_testDivision.IccbbaRegistrationNumber = _iccbbaRegistrationNumber;
			Assert.AreEqual(_iccbbaRegistrationNumber,_testDivision.IccbbaRegistrationNumber);

		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void ICCBBARegistrationNumber_Set_Fail()
		{
			string _iccbbaRegistrationNumber = string.Empty.PadRight(5,'X');
			_testDivision.IccbbaRegistrationNumber = _iccbbaRegistrationNumber;
		}

		[Test]
		public void LocalSupplierGuid_Pass()
		{
			Guid _localSupplierGUID = _testDivision.LocalSupplierGuid;
			Assert.AreEqual(_testDivision.LocalSupplierGuid,_localSupplierGUID);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void LocalSupplierGuid_Fail()
		{
		}

		[Test]
		public void BloodProductModifications_Get()
		{
			BOL.BloodProductModifications _bloodProductModifications = _testDivision.BloodProductModifications;
			Assert.AreEqual(_testDivision.BloodProductModifications,_bloodProductModifications);
		}

		[Test]
		public void BloodProductModifications_Set()
		{
			BOL.BloodProductModifications _bloodProductModifications = new BOL.BloodProductModifications();
			_testDivision.BloodProductModifications = _bloodProductModifications;
			Assert.AreEqual(_bloodProductModifications,_testDivision.BloodProductModifications);
			//
			_bloodProductModifications = new BOL.BloodProductModifications();
			_testDivision.BloodProductModifications = _bloodProductModifications;
			Assert.AreEqual(_bloodProductModifications,_testDivision.BloodProductModifications);
		}


		[Test]
		public void AutomatedInstrumentIndicator_Pass()
		{
			bool _automatedInstrumentIndicator = true;
			_testDivision.AutomatedInstrumentIndicator = _automatedInstrumentIndicator;
			Assert.AreEqual(_automatedInstrumentIndicator,_testDivision.AutomatedInstrumentIndicator);

			_automatedInstrumentIndicator = false;
			_testDivision.AutomatedInstrumentIndicator = _automatedInstrumentIndicator;
			Assert.AreEqual(_automatedInstrumentIndicator,_testDivision.AutomatedInstrumentIndicator);
			
			Assert.AreEqual(_testDivision.IsDirty, true);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AutomatedInstrumentIndicator_Fail()
		{
		}

		[Test]
		public void PatientTestingConfiguration_Pass()
		{
			Array patientTestingConfigurations = Enum.GetValues(typeof(Common.PatientTestingConfiguration));

			foreach (Common.PatientTestingConfiguration patientTestingConfiguration in patientTestingConfigurations)
			{
				_testDivision.PatientTestingConfiguration = patientTestingConfiguration;
				Common.PatientTestingConfiguration _patientTestingConfiguration = _testDivision.PatientTestingConfiguration;
				Assert.AreEqual(_testDivision.PatientTestingConfiguration,_patientTestingConfiguration);
			}
			
			Assert.AreEqual(_testDivision.IsDirty, true);			
		}

				
		[Test]
		public void PatientTestingConfiguration_Fail()
		{
			this._testDivision.SetInitialBrokenRules(Common.VbecsBrokenRules.DlgMaintainDivision.FormClassName, Common.VbecsBrokenRules.DlgMaintainDivision.RuleSets.Alerts);
            
			this._testDivision.PatientTestingConfiguration = Common.PatientTestingConfiguration.Unknown;
			Assert.AreEqual(true, this._testDivision.IsRuleBroken(Common.VbecsBrokenRules.DlgMaintainDivision.PatientTestingConfiguration), "BrokenRule");
		}

		[Test]
		public void MaximumSpecimenExpirationDays_Pass()
		{
			short _maximumSpecimenExpirationDays = 30;
			_testDivision.MaximumSpecimenExpirationDays = _maximumSpecimenExpirationDays;
			Assert.AreEqual(_maximumSpecimenExpirationDays,_testDivision.MaximumSpecimenExpirationDays);			
		}

		[Test]
		public void MaximumSpecimenExpirationDays_Fail()
		{
			this._testDivision.MaximumSpecimenExpirationDays = Convert.ToInt16(94);
			Assert.IsTrue(this._testDivision.IsRuleBroken(BROKEN.MaximumSpecimenExpirationDays));

			this._testDivision.MaximumSpecimenExpirationDays = Convert.ToInt16(2);
			Assert.IsTrue(this._testDivision.IsRuleBroken(BROKEN.MaximumSpecimenExpirationDays));
		}

		
		[Test]
		public void DivisionAlerts_Pass()
		{
			BOL.DivisionAlerts tstValue = new BOL.DivisionAlerts(this._drDivision);

			this._testDivision.DivisionAlerts = tstValue;

			Assert.AreEqual(this._testDivision.DivisionAlerts, tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void DivisionAlerts_Fail()
		{
		}

		
		[Test]
		public void MedicalDirectorName_Pass()
		{
			string tstValue = System.DateTime.Now.Date.ToString();

			this._testDivision.MedicalDirectorName = tstValue;

			Assert.AreEqual(this._testDivision.MedicalDirectorName, tstValue);			
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void MedicalDirectorName_Fail()
		{
			this._testDivision.MedicalDirectorName = "".PadRight(31,'X');
		}

		[Test]
		public void GetMedicalDirectors_Pass()
		{
			string [] directors = BOL.Division.GetMedicalDirectors();

			Assert.IsNotNull(directors);
			Assert.IsTrue(directors.Length>0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetMedicalDirectors_Fail()
		{
		}

		[Test]
		public void Address_Pass()
		{
			BOL.Address address=this._testDivision.Address; 

			Assert.IsNotNull(address);
			Assert.IsNotNull(address.AddressGuid);
			Assert.IsTrue(address.StateCode.Length>0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void Address_Fail()
		{
		}

	}
}
#endif


